/**
 * blogs.js
 * Handles:
 * 1. Blog data structure with unique images.
 * 2. Dynamic generation of blog cards into the HTML grid.
 * 3. IntersectionObserver for card reveal animations.
 * 4. Modal open/close, dynamic content loading, and scroll lock.
 */

document.addEventListener('DOMContentLoaded', () => {

    // --- 1. Blog Data Structure ---
    const blogs = [
        {
            id: 1,
            title: "Exposing the Enemy Within: A Spiritual Insight into Betrayal",
            author: "The Editorial Team",
            date: "March 31, 2024",
            image: "assets/img/expose.jpg",
            excerpt: "Delving into the intricate dynamics of betrayal and deception, it becomes evident that the enemy often operates from within, cunningly disguised amidst trusted networks.",
            content: `
                <p>Betrayal is one of the most painful experiences in the human journey, yet spiritually, it is a recurrent theme. The enemy's oldest tactic is to sow discord not from the outside, but from within the ranks, targeting close relationships and trusted circles.</p>
                <p>Understanding this spiritual reality allows us to guard our hearts with wisdom. The 'enemy within' is often not a person, but the spirit of offense, jealousy, or insecurity that finds fertile ground in human frailty.</p>
                <blockquote>
                    "The Lord protects the faithful, but the proud he pays back in full." — Psalm 31:23 (NIV)
                </blockquote>
                <p>We are called to pray for discernment, to forgive quickly, and to anchor our trust in Christ alone, who perfectly understands betrayal yet overcame it through love.</p>
            `
        },
        {
            id: 2,
            title: "The Holy Spirit: Our Guide to Expansion",
            author: "Pastor Daniel",
            date: "February 12, 2024",
            image: "assets/img/guide.jpg",
            excerpt: "Understanding Our Divine Design: In Genesis 3:22, we are reminded that when the Lord created us, He did so in His image and likeness. This article guides us to expansion through the Spirit.",
            content: `
                <p>The Holy Spirit is not merely an optional extra in the Christian life; He is the very essence of our power and the blueprint for our expansion. Created in God's image, we possess a divine capacity that is fully unlocked only through His guidance.</p>
                <p>Expansion is not just about size, but about influence, depth, and reach. The Spirit guides our decisions, empowers our witness, and connects us with divine opportunities that stretch our capacity beyond the natural.</p>
                <p>To experience true expansion, we must yield completely to the Holy Spirit's direction, allowing Him to prune the old season and usher us into new territories of grace and ministry.</p>
            `
        },
        {
        id: 3,
            title: "A Joyous Celebration: 40 Baptised in Our Church",
            author: "The Church Leadership",
            date: "February 10, 2024",
            image: "assets/img/BAPTISM.JPG",
            excerpt: "In a profound display of faith and commitment, our church recently witnessed the baptism of 40 individuals, marking a significant milestone in their spiritual journey.",
            content: `
                <p>The baptism service was a powerful reminder of God's ongoing work in our midst. Forty individuals publicly declared their commitment to Christ, symbolizing their death to the old life and resurrection into a new life in Christ.</p>
                <p>We celebrated with immense joy as these brothers and sisters took the plunge, embracing this foundational ordinance. It was a visible manifestation of the grace and transformation that is happening daily within our congregation.</p>
                <p>We continue to pray for these new members, that they walk in the fullness of their calling and experience the boundless joy of following Jesus.</p>
            `
        },
        {
            id: 4,
            title: "Navigating the Prophetic Calling: Avoiding Pitfalls and Pursuing Purpose",
            author: "The Editorial Team",
            date: "February 9, 2024",
            image: "assets/img/Calling.jpg",
            excerpt: "The prophetic calling stands as one of the most unique ministries within the body of Christ. As believers, we aspire for all to operate with integrity and purpose.",
            content: `
                <p>The prophetic gift, while essential, requires careful stewardship. Navigating this calling means prioritizing character over gifts, ensuring every word aligns with scripture, and submitting to spiritual accountability.</p>
                <p>The pitfalls are pride and inaccuracy. We must continually check our motives and ensure we are speaking God's heart, not our own opinions or desires. The goal is always to build up, encourage, and comfort (1 Corinthians 14:3).</p>
                <p>Pursuing purpose means faithfully delivering the messages entrusted to you, without fear of man or desire for applause. Integrity is the foundation of genuine prophetic ministry.</p>
            `
        },
    ];

    // --- 2. Dynamic Card Generation ---
    const postsContainer = document.getElementById('posts-container');

    const createBlogCard = (blog) => {
        const article = document.createElement('article');
        article.classList.add('blog-card', 'reveal-card');
        article.setAttribute('data-blog-id', blog.id);

        // Uses a lower resolution for the card view for better performance
        const cardImageSrc = blog.image.replace('800/400', '768/384');

        article.innerHTML = `
            <img src="${cardImageSrc}" alt="Image for ${blog.title}" loading="lazy">
            <div class="card-content">
                <h3 class="heading-serif color-purple">${blog.title}</h3>
                <p class="card-meta color-purple-soft">
                    <span class="card-date"><i class="fas fa-calendar-alt"></i> ${blog.date}</span> 
                    <span class="card-comments"><i class="fas fa-comments"></i> No Comments</span> 
                </p>
                <p class="card-excerpt">${blog.excerpt}</p>
                <button class="btn btn-outline-purple read-more-btn" data-blog-id="${blog.id}" aria-controls="blog-modal">Read More »</button>
            </div>
        `;
        return article;
    };

    const initializeGrid = () => {
        postsContainer.innerHTML = '';
        blogs.forEach(blog => {
            const card = createBlogCard(blog);
            postsContainer.appendChild(card);
        });

        // Re-assign listeners after cards are generated
        attachReadMoreListeners();
        // Re-run observer on new elements
        document.querySelectorAll('.reveal-card').forEach(el => observer.observe(el));
    };


    // --- 3. IntersectionObserver for Card Reveal Animations ---
    const observerOptions = { root: null, rootMargin: '0px', threshold: 0.1 };

    const observer = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const element = entry.target;
                
                // For Hero/Sidebar animations
                if (element.classList.contains('slide-up')) {
                    element.classList.add('visible');
                } else if (element.classList.contains('reveal-card')) {
                    element.classList.add('visible');
                }
                observer.unobserve(element);
            }
        });
    }, observerOptions);

    // Initial observer targets (Sections, Sidebar)
    document.querySelectorAll('.slide-up, .hero-section .fade-in').forEach(el => observer.observe(el));


    // --- 4. Modal Functionality ---
    const modal = document.getElementById('blog-modal');
    const modalCloseBtn = modal.querySelector('.modal-close-btn');

    const openModal = (blogId) => {
        const data = blogs.find(blog => blog.id === parseInt(blogId));
        if (!data) return;

        // Populate modal content
        document.getElementById('modal-title').textContent = data.title;
        document.getElementById('modal-author').innerHTML = `<i class="fas fa-user-circle"></i> ${data.author}`;
        document.getElementById('modal-date').innerHTML = `<i class="fas fa-calendar-alt"></i> ${data.date}`;
        // Full size image for the modal
        document.getElementById('modal-image').setAttribute('src', data.image);
        document.getElementById('modal-image').setAttribute('alt', `Featured image for ${data.title}`);
        document.getElementById('modal-content').innerHTML = data.content;
        
        // Show modal and lock scroll
        modal.classList.add('is-open');
        modal.setAttribute('aria-hidden', 'false');
        document.body.style.overflow = 'hidden';
        
        modal.querySelector('.modal-content').scrollTop = 0;
    };

    const closeModal = () => {
        modal.classList.remove('is-open');
        modal.setAttribute('aria-hidden', 'true');
        document.body.style.overflow = ''; // Restore body scrolling
    };

    const attachReadMoreListeners = () => {
        document.querySelectorAll('.read-more-btn').forEach(button => {
            button.addEventListener('click', function() {
                const blogId = this.getAttribute('data-blog-id');
                openModal(blogId);
            });
        });
    };

    // Global Listeners
    modalCloseBtn.addEventListener('click', closeModal);
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            closeModal();
        }
    });
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' && modal.classList.contains('is-open')) {
            closeModal();
        }
    });

    // Initial setup
    initializeGrid();
});