/**
 * events.js
 * Handles:
 * 1. IntersectionObserver for reveal-on-scroll.
 * 2. Event filtering (Upcoming/Past/All).
 * 3. Modal functionality for event details.
 */

document.addEventListener('DOMContentLoaded', () => {

    // --- 1. Event Data (Mock Data for Modal) ---
    const eventData = {
        1: {
            title: "Annual Prayer Summit",
            date: "October 25, 2025 | 7:00 PM - 9:00 PM",
            location: "Main Sanctuary, 123 Church Rd, City",
            description: "Join us for our most impactful evening of the year. This is a dedicated time for corporate prayer, intercession for the community, and personal spiritual breakthrough. Childcare will be provided.",
            registerLink: "#register-prayer-summit"
        },
        2: {
            title: "Women's Fellowship Brunch",
            date: "November 9, 2025 | 10:00 AM - 12:00 PM",
            location: "Church Hall (Dress Code: Elegant Casual)",
            description: "A beautiful gathering of women for fellowship, inspiration, and delicious food. Our guest speaker, Sister Eleanor Vance, will be discussing 'Finding Strength in Silence'. Tickets required.",
            registerLink: "#register-women-brunch"
        },
        3: {
            title: "Summer Youth Camp",
            date: "July 2025 | 5 Days (View Memories)",
            location: "Camp Cedar Creek",
            description: "This event is completed. View photos and testimonials from the most dynamic week of the summer, where our youth deepened their faith through adventure and worship.",
            registerLink: "#view-photo-gallery"
        },
        4: {
            title: "Community Outreach Day",
            date: "December 14, 2025 | 9:00 AM - 3:00 PM",
            location: "City Park (Meet at Church at 8:30 AM)",
            description: "Serve the local community! We will be distributing food and supplies, performing light cleanup, and sharing the love of Christ. Volunteers needed—sign up early!",
            registerLink: "#volunteer-outreach"
        }
    };


    // --- 2. IntersectionObserver for Reveal Animations ---
    const observerOptions = {
        root: null, // viewport
        rootMargin: '0px',
        threshold: 0.1 
    };

    const observer = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const element = entry.target;
                
                // Handle staggered elements (hero, cta)
                if (element.classList.contains('reveal-stagger')) {
                    element.querySelectorAll('.fade-in, .slide-up').forEach(staggerEl => {
                        const delay = parseFloat(staggerEl.getAttribute('data-delay')) * 1000 || 0;
                        setTimeout(() => staggerEl.classList.add('visible'), delay);
                    });
                } 
                
                // Handle individual cards and timeline items
                else if (element.classList.contains('reveal-card') || element.classList.contains('timeline-item')) {
                    element.classList.add('visible');
                }

                observer.unobserve(element);
            }
        });
    }, observerOptions);

    // Observe all relevant elements
    document.querySelectorAll('.reveal-stagger, .reveal-card, .timeline-item').forEach(el => {
        observer.observe(el);
    });

    // --- 3. Event Filtering Logic ---
    const filterButtons = document.querySelectorAll('.filter-btn');
    const eventCards = document.querySelectorAll('.event-card');

    const filterEvents = (filterCategory) => {
        eventCards.forEach(card => {
            const category = card.getAttribute('data-category');
            const isVisible = (filterCategory === 'all' || category === filterCategory);
            
            // Use opacity for smooth transition
            card.style.opacity = isVisible ? '1' : '0';
            card.style.display = isVisible ? 'block' : 'none'; // Use display:none for true filtering
        });
    };

    filterButtons.forEach(button => {
        button.addEventListener('click', function() {
            // Update active state
            filterButtons.forEach(btn => btn.classList.remove('is-active'));
            this.classList.add('is-active');

            // Apply filter
            const filterCategory = this.getAttribute('data-filter');
            filterEvents(filterCategory);
        });
    });

    // Run initial filter (Upcoming is active by default)
    filterEvents('upcoming');


    // --- 4. Modal Functionality ---
    const modal = document.getElementById('event-modal');
    const modalCloseBtn = modal.querySelector('.modal-close-btn');
    const learnMoreButtons = document.querySelectorAll('.learn-more-btn');

    const openModal = (eventId) => {
        const data = eventData[eventId];
        if (!data) return;

        // Populate modal content
        document.getElementById('modal-title').textContent = data.title;
        document.getElementById('modal-date').textContent = data.date;
        document.getElementById('modal-location').textContent = data.location;
        document.getElementById('modal-description').textContent = data.description;
        
        const registerBtn = document.getElementById('modal-register-btn');
        registerBtn.setAttribute('href', data.registerLink);
        
        // Show modal
        modal.classList.add('is-open');
        modal.setAttribute('aria-hidden', 'false');
    };

    const closeModal = () => {
        modal.classList.remove('is-open');
        modal.setAttribute('aria-hidden', 'true');
    };

    // Attach click listeners to "Learn More" buttons
    learnMoreButtons.forEach(button => {
        button.addEventListener('click', function() {
            const eventId = this.getAttribute('data-event-id');
            openModal(eventId);
        });
    });

    // Close modal listeners
    modalCloseBtn.addEventListener('click', closeModal);
    
    // Close modal if user clicks outside of it
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            closeModal();
        }
    });

    // Close modal if user presses ESC key
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' && modal.classList.contains('is-open')) {
            closeModal();
        }
    });

});