/**
 * gallery.js
 * Handles: Dynamic Image Population, IntersectionObserver, Filtering, Load More, and Lightbox.
 */

document.addEventListener('DOMContentLoaded', () => {

    // --- DOM Element References ---
    const galleryGrid = document.getElementById('galleryGrid');
    const filterButtons = document.querySelectorAll('.filter-btn');
    const loadMoreBtn = document.getElementById('loadMoreBtn');
    const lightboxModal = document.getElementById('lightboxModal');
    const lightboxImage = document.getElementById('lightboxImage');
    const lightboxCaption = document.getElementById('lightboxCaption');
    const lightboxClose = document.getElementById('lightboxClose');
    const lightboxPrev = document.getElementById('lightboxPrev');
    const lightboxNext = document.getElementById('lightboxNext');

    // --- State Variables ---
    let currentFilter = 'all';
    let currentIndex = 0; // Index of the last loaded image in the filtered/current view
    const itemsPerLoad = 9;
    let visibleItems = []; // Array of image data objects currently visible
    let allImagesData = []; // Array of image data objects matching the current filter

    // --- Mock Data (Enhanced) ---
    const MOCK_IMAGES = [
        { id: 1, src: "https://via.placeholder.com/600x400/2C0033/D4AF37?text=Sunday+Worship", alt: "Congregation during a powerful worship service, hands raised.", caption: "Hands raised in joyful worship.", category: "worship" },
        { id: 2, src: "https://via.placeholder.com/600x400/4B1A47/FAF8F6?text=Youth+Trip+1", alt: "Youth group smiling together at an outdoor retreat.", caption: "Building faith and friendship.", category: "youth" },
        { id: 3, src: "https://via.placeholder.com/600x400/D4AF37/2C0033?text=Outreach+Day+1", alt: "Volunteers serving meals at a local community center.", caption: "Serving our community with love.", category: "outreach" },
        { id: 4, src: "https://via.placeholder.com/600x400/FAF8F6/2C0033?text=Conference+Speaker", alt: "Pastor delivering a sermon at a leadership conference.", caption: "Inspiring words at the annual conference.", category: "conferences" },
        { id: 5, src: "https://via.placeholder.com/600x400/2C0033/D4AF37?text=Baptism+Moment", alt: "A joyful moment of baptism in the sanctuary.", caption: "New life in Christ.", category: "worship" },
        { id: 6, src: "https://via.placeholder.com/600x400/4B1A47/FAF8F6?text=Family+Event", alt: "Families enjoying games at a church fun day.", caption: "Our annual family fellowship event.", category: "events" },
        { id: 7, src: "https://via.placeholder.com/600x400/D4AF37/2C0033?text=Bible+Study", alt: "Small group gathering for a mid-week Bible study.", caption: "Deepening our knowledge of scripture.", category: "worship" },
        { id: 8, src: "https://via.placeholder.com/600x400/FAF8F6/2C0033?text=Teen+Activity", alt: "Teenagers collaborating on a group project.", caption: "Engaging teens ministry.", category: "youth" },
        { id: 9, src: "https://via.placeholder.com/600x400/2C0033/D4AF37?text=Community+CleanUp", alt: "Church members participating in a community cleanup.", caption: "Making a difference together.", category: "outreach" },
        { id: 10, src: "https://via.placeholder.com/600x400/4B1A47/FAF8F6?text=Worship+Team", alt: "Worship team leading the congregation in song.", caption: "Leading with spirit.", category: "worship" },
        { id: 11, src: "https://via.placeholder.com/600x400/D4AF37/2C0033?text=Graduation+Celebration", alt: "Church members celebrating recent graduates.", caption: "Celebrating milestones.", category: "events" },
        { id: 12, src: "https://via.placeholder.com/600x400/FAF8F6/2C0033?text=Women's+Retreat", alt: "Women gathered outdoors for a spiritual retreat.", caption: "Time for reflection and growth.", category: "conferences" },
        // Add more mock data to enable 'Load More' to work
        { id: 13, src: "https://via.placeholder.com/600x400/2C0033/D4AF37?text=Fellowship+Lunch", alt: "Congregation having lunch after service.", caption: "Post-service fellowship lunch.", category: "events" },
        { id: 14, src: "https://via.placeholder.com/600x400/4B1A47/FAF8F6?text=Choir+Rehearsal", alt: "Choir members practicing for Sunday service.", caption: "Practicing harmonious praise.", category: "worship" },
        { id: 15, src: "https://via.placeholder.com/600x400/D4AF37/2C0033?text=Mission+Kids", alt: "Volunteers teaching children on a mission trip.", caption: "Teaching the next generation.", category: "outreach" },
    ];


    // --- 1. IntersectionObserver for Reveal Animation ---
    const galleryObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('visible');
                observer.unobserve(entry.target);
            }
        });
    }, { root: null, rootMargin: '0px', threshold: 0.1 });

    function observeNewItems() {
        // Observe gallery items and top-level animations
        document.querySelectorAll('.gallery-item:not(.visible), .fade-in, .slide-up').forEach(el => {
            // Apply staggered delay to top-level elements if defined
            const delayAttr = el.getAttribute('data-delay');
            if (delayAttr) {
                 const delay = parseFloat(delayAttr.replace('s', '')) * 1000;
                 setTimeout(() => galleryObserver.observe(el), delay);
            } else {
                 galleryObserver.observe(el);
            }
        });
    }


    // --- 2. Gallery Building & Loading Functions ---
    function createGalleryItem(image, index) {
        const item = document.createElement('figure');
        item.className = 'gallery-item';
        item.setAttribute('data-category', image.category);
        item.setAttribute('data-index', index);
        item.tabIndex = 0;
        item.setAttribute('role', 'listitem');

        item.innerHTML = `
            <img src="${image.src}" alt="${image.alt}" loading="lazy">
            <figcaption class="caption-overlay">${image.caption}</figcaption>
        `;

        item.addEventListener('click', () => openLightbox(index));
        item.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                openLightbox(index);
            }
        });
        return item;
    }

    function renderGallery(images) {
        galleryGrid.innerHTML = '';
        images.forEach((image, index) => {
            galleryGrid.appendChild(createGalleryItem(image, index));
        });
        
        visibleItems = images;
        currentIndex = visibleItems.length; // Next item to load
        observeNewItems();
        checkLoadMoreButton();
    }

    function loadInitialGallery() {
        // Filter the full dataset initially
        allImagesData = MOCK_IMAGES.filter(img => currentFilter === 'all' || img.category === currentFilter);
        
        const initialLoad = allImagesData.slice(0, itemsPerLoad);
        renderGallery(initialLoad);
    }

    function loadMoreImages() {
        const nextBatch = allImagesData.slice(currentIndex, currentIndex + itemsPerLoad);
        if (nextBatch.length === 0) return; // Nothing left to load

        nextBatch.forEach((image) => {
            const item = createGalleryItem(image, currentIndex);
            galleryGrid.appendChild(item);
            visibleItems.push(image);
            currentIndex++;
        });

        observeNewItems();
        checkLoadMoreButton();
    }

    function checkLoadMoreButton() {
        if (currentIndex >= allImagesData.length) {
            loadMoreBtn.style.display = 'none';
        } else {
            loadMoreBtn.style.display = 'inline-block';
        }
    }


    // --- 3. Filtering Logic ---
    function filterGallery(filter) {
        currentFilter = filter;

        filterButtons.forEach(btn => {
            if (btn.getAttribute('data-filter') === filter) {
                btn.classList.add('active');
            } else {
                btn.classList.remove('active');
            }
        });

        // Re-filter the full data set
        allImagesData = MOCK_IMAGES.filter(img => filter === 'all' || img.category === filter);

        // Reset and re-render the first batch
        currentIndex = 0;
        const initialLoad = allImagesData.slice(0, itemsPerLoad);
        renderGallery(initialLoad);
    }

    filterButtons.forEach(button => {
        button.addEventListener('click', (e) => {
            filterGallery(e.target.getAttribute('data-filter'));
        });
    });

    loadMoreBtn.addEventListener('click', loadMoreImages);


    // --- 4. Lightbox Functionality ---

    function updateLightboxContent(index) {
        const image = visibleItems[index];
        lightboxImage.src = image.src;
        lightboxImage.alt = image.alt;
        lightboxCaption.textContent = image.caption;
        
        // Update nav buttons visibility
        lightboxPrev.style.display = index > 0 ? 'flex' : 'none';
        lightboxNext.style.display = index < visibleItems.length - 1 ? 'flex' : 'none';

        currentIndex = index;
    }

    function openLightbox(index) {
        updateLightboxContent(index);
        lightboxModal.classList.add('active');
        lightboxModal.setAttribute('aria-hidden', 'false');
        document.body.style.overflow = 'hidden';
        lightboxClose.focus();
    }

    function closeLightbox() {
        lightboxModal.classList.remove('active');
        lightboxModal.setAttribute('aria-hidden', 'true');
        document.body.style.overflow = '';
        
        // Find and re-focus the corresponding gallery item for accessibility
        const itemToRefocus = document.querySelector(`.gallery-item[data-index="${currentIndex}"]`);
        if (itemToRefocus) itemToRefocus.focus();
    }

    function showPrev() {
        if (currentIndex > 0) {
            updateLightboxContent(currentIndex - 1);
        }
    }

    function showNext() {
        if (currentIndex < visibleItems.length - 1) {
            updateLightboxContent(currentIndex + 1);
        }
    }

    // Lightbox Event Listeners
    lightboxClose.addEventListener('click', closeLightbox);
    lightboxPrev.addEventListener('click', showPrev);
    lightboxNext.addEventListener('click', showNext);
    
    // Close on click outside the image
    lightboxModal.addEventListener('click', (e) => {
        if (e.target === lightboxModal) {
            closeLightbox();
        }
    });

    // Keyboard navigation (Esc, Left/Right Arrows)
    document.addEventListener('keydown', (e) => {
        if (lightboxModal.classList.contains('active')) {
            if (e.key === 'Escape') {
                closeLightbox();
            } else if (e.key === 'ArrowLeft') {
                showPrev();
            } else if (e.key === 'ArrowRight') {
                showNext();
            }
        }
    });

    // --- Initialization ---
    loadInitialGallery();
    // Observe initial static elements (Hero/Filter)
    observeNewItems(); 
});