/**
 * give.js
 * Handles:
 * 1. IntersectionObserver for reveal animations.
 * 2. Dynamic event listeners for giving buttons.
 * 3. Modal open/close logic.
 * 4. Smooth scrolling to the online form.
 * 5. Form submission handling with a confirmation message.
 */

document.addEventListener('DOMContentLoaded', () => {

    // --- Aesthetic Data for Modal ---
    const givingMethods = {
        online: {
            title: "Online Giving",
            text: "You will be redirected to our secure third-party payment portal. Please confirm to proceed.",
            action: () => { 
                // In a real app, this would open the payment link/page
                console.log("Redirecting to online payment...");
                window.location.href = "#online-form"; // Scroll to the form on this page
            }
        },
        bank: {
            title: "Bank Transfer Details",
            text: "All necessary bank details are provided below this section. Please ensure you include your Name and Fund Type (e.g., Tithe) in the reference.",
            action: () => { 
                // Scrolls to bank details section
                const bankDetailsSection = document.getElementById('bank-details');
                bankDetailsSection.scrollIntoView({ behavior: 'smooth' });
                closeModal();
            }
        },
        mobile: {
            title: "Mobile Money / Cash App",
            text: "Please send your gift to the official church number: +123 456 7890. Use 'Partnership' or 'Tithe' as the reference. Call if you require further assistance.",
            action: () => { console.log("User viewing Mobile Money instructions."); }
        },
        inperson: {
            title: "In-Person Giving",
            text: "You can drop your tithes and offerings during any of our physical service times. Visit our main page for service schedules.",
            action: () => { console.log("User viewing In-Person instructions."); }
        }
    };

    // --- 1. Intersection Observer for Animations ---
    const observerOptions = { root: null, rootMargin: '0px', threshold: 0.1 };

    const observer = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const element = entry.target;
                element.classList.add('visible');
                
                // Handle staggered internal elements (like the hero text)
                if (element.id === 'hero' || element.id === 'impact') {
                    element.querySelectorAll('.fade-in').forEach(staggerEl => {
                        const delay = parseFloat(staggerEl.getAttribute('data-delay')) * 1000 || 0;
                        setTimeout(() => staggerEl.classList.add('visible'), delay);
                    });
                }
                observer.unobserve(element);
            }
        });
    }, observerOptions);

    // Apply observer to all revealable sections and elements
    document.querySelectorAll('.reveal-fade, .reveal-slide').forEach(el => observer.observe(el));


    // --- 2 & 3. Modal Functionality ---
    const modal = document.getElementById('giving-modal');
    const modalTitle = document.getElementById('modal-title');
    const modalText = document.getElementById('modal-text');
    const modalActionBtn = document.getElementById('modal-action-btn');
    const modalCloseBtn = document.getElementById('modal-close-btn');
    const closeBtnX = modal.querySelector('.modal-close-btn');

    const openModal = (method) => {
        const data = givingMethods[method];
        if (!data) return;

        modalTitle.textContent = data.title;
        modalText.textContent = data.text;
        
        // Update action button based on method
        modalActionBtn.textContent = method === 'bank' ? "Go to Details" : (method === 'online' ? "Proceed to Form" : "Understood");
        modalActionBtn.onclick = () => {
            data.action();
            if (method !== 'online' && method !== 'bank') {
                closeModal(); // Close modal after action for static methods
            }
        };

        modal.classList.add('is-open');
        modal.setAttribute('aria-hidden', 'false');
        document.body.style.overflow = 'hidden';
    };

    const closeModal = () => {
        modal.classList.remove('is-open');
        modal.setAttribute('aria-hidden', 'true');
        document.body.style.overflow = ''; // Restore body scrolling
    };
    
    // Attach listeners for giving option buttons
    document.querySelectorAll('.give-button').forEach(button => {
        button.addEventListener('click', function() {
            const method = this.getAttribute('data-method');
            if (method === 'online') {
                 // For online, immediately scroll to the form section
                document.getElementById('online-form').scrollIntoView({ behavior: 'smooth' });
            } else {
                openModal(method);
            }
        });
    });

    // Modal close listeners
    modalCloseBtn.addEventListener('click', closeModal);
    closeBtnX.addEventListener('click', closeModal);
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            closeModal();
        }
    });
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' && modal.classList.contains('is-open')) {
            closeModal();
        }
    });


    // --- 4. Form Submission Handling (Basic Validation & Toast Simulation) ---
    const form = document.getElementById('donation-form');

    const showToast = (message, isError = false) => {
        // Create a simple styled toast message (instead of alert)
        const toast = document.createElement('div');
        toast.textContent = message;
        toast.style.cssText = `
            position: fixed; bottom: 20px; right: 20px; 
            padding: 15px 30px; border-radius: 8px; 
            color: var(--color-text-light); z-index: 1001;
            font-weight: 500;
            background-color: ${isError ? 'red' : 'green'};
            opacity: 0; transition: opacity 0.5s ease;
            box-shadow: 0 4px 10px rgba(0,0,0,0.3);
        `;
        document.body.appendChild(toast);
        
        // Fade in
        setTimeout(() => { toast.style.opacity = 1; }, 10);
        
        // Fade out and remove
        setTimeout(() => { toast.style.opacity = 0; }, 3000);
        setTimeout(() => { toast.remove(); }, 3500);
    };

    form.addEventListener('submit', function(event) {
        event.preventDefault();
        
        // Basic validation check
        const amount = document.getElementById('amount').value;
        const email = document.getElementById('email').value;
        const fund = document.getElementById('fund').value;

        if (!amount || !email || !fund) {
             showToast("Please fill out all required fields.", true);
             return;
        }

        // Simulate form processing/redirection to payment gateway
        console.log("Form Data Submitted:", {
            amount: amount,
            email: email,
            fund: fund,
            name: document.getElementById('name').value,
            message: document.getElementById('message').value
        });

        // Clear form and show success toast
        form.reset();
        showToast("Thank you for your generosity! Redirecting to payment...");
    });
});