/**
 * ministries.js
 * Handles:
 * 1. IntersectionObserver for reveal-on-scroll (fade/slide).
 * 2. Full Carousel functionality (navigation, dots, sliding, resize).
 */

document.addEventListener('DOMContentLoaded', () => {

    // --- 1. IntersectionObserver for Reveal Animations ---
    const observerOptions = {
        root: null, // viewport
        rootMargin: '0px',
        threshold: 0.1 
    };

    const observer = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const element = entry.target;
                
                // Handle staggered animation for hero/cta
                if (element.classList.contains('reveal-stagger')) {
                    element.querySelectorAll('.fade-in, .slide-up').forEach(staggerEl => {
                        // Apply staggered delay (if set in HTML)
                        const delay = parseFloat(staggerEl.getAttribute('data-delay')) * 1000 || 0;
                        setTimeout(() => staggerEl.classList.add('visible'), delay);
                    });
                } 
                
                // Handle the main ministry blocks
                else if (element.classList.contains('reveal-block')) {
                    element.classList.add('visible');
                }

                observer.unobserve(element); // Stop observing once visible
            }
        });
    }, observerOptions);

    // Observe containers that hold the staggered elements and the ministry blocks
    document.querySelectorAll('.reveal-stagger, .reveal-block').forEach(el => {
        observer.observe(el);
    });

    // --- 2. Carousel Functionality ---

    const carousels = document.querySelectorAll('.carousel');

    carousels.forEach(carousel => {
        const track = carousel.querySelector('.carousel-track');
        const slides = Array.from(track.children);
        const nextButton = carousel.querySelector('.carousel-button.next');
        const prevButton = carousel.querySelector('.carousel-button.prev');
        const dotsNav = carousel.querySelector('.carousel-nav-dots');
        const dots = Array.from(dotsNav.children);

        if (!slides.length) return;
        
        // Initial Calculation (Must run before setting positions)
        let slideSize = slides[0].getBoundingClientRect().width;

        // Set initial positions for all slides
        const setSlidePosition = (slide, index) => {
            slide.style.left = slideSize * index + 'px';
        };
        slides.forEach(setSlidePosition);

        // Utility function to get the current active slide
        const getActiveSlide = () => track.querySelector('.is-active');
        
        // Utility function to get the index of a slide
        const getSlideIndex = (slide) => slides.findIndex(s => s === slide);

        // Function to move the track
        const moveToSlide = (currentSlide, targetSlide) => {
            const amountToMove = targetSlide.style.left;
            track.style.transform = 'translateX(-' + amountToMove + ')';
            
            // Update active state
            currentSlide.classList.remove('is-active');
            currentSlide.setAttribute('aria-hidden', 'true');
            targetSlide.classList.add('is-active');
            targetSlide.setAttribute('aria-hidden', 'false');
        };

        // Function to update the dot navigation
        const updateDots = (currentDot, targetDot) => {
            currentDot.classList.remove('is-active');
            targetDot.classList.add('is-active');
        };

        // Function to show/hide the navigation arrows
        const updateArrows = (targetIndex) => {
            prevButton.classList.toggle('is-hidden', targetIndex === 0);
            nextButton.classList.toggle('is-hidden', targetIndex === slides.length - 1);
        };

        // --- Event Listeners ---

        // 1. Click on Next button
        nextButton.addEventListener('click', () => {
            const currentSlide = getActiveSlide();
            const nextSlide = currentSlide.nextElementSibling;
            if (!nextSlide) return;

            const currentDot = dotsNav.querySelector('.is-active');
            const nextDot = currentDot.nextElementSibling;
            const nextIndex = getSlideIndex(nextSlide);

            moveToSlide(currentSlide, nextSlide);
            updateDots(currentDot, nextDot);
            updateArrows(nextIndex);
        });

        // 2. Click on Previous button
        prevButton.addEventListener('click', () => {
            const currentSlide = getActiveSlide();
            const prevSlide = currentSlide.previousElementSibling;
            if (!prevSlide) return;
            
            const currentDot = dotsNav.querySelector('.is-active');
            const prevDot = currentDot.previousElementSibling;
            const prevIndex = getSlideIndex(prevSlide);

            moveToSlide(currentSlide, prevSlide);
            updateDots(currentDot, prevDot);
            updateArrows(prevIndex);
        });

        // 3. Click on the dot indicators
        dotsNav.addEventListener('click', e => {
            const targetDot = e.target.closest('button');

            if (!targetDot) return;

            const currentDot = dotsNav.querySelector('.is-active');
            const targetIndex = dots.findIndex(dot => dot === targetDot);
            const targetSlide = slides[targetIndex];

            // Only move if clicking a non-active dot
            if (!targetDot.classList.contains('is-active')) {
                moveToSlide(getActiveSlide(), targetSlide);
                updateDots(currentDot, targetDot);
                updateArrows(targetIndex);
            }
        });

        // 4. Handle window resize (Recalculate slide positions)
        window.addEventListener('resize', () => {
            // Recalculate size
            slideSize = slides[0].getBoundingClientRect().width;

            // Update all slide positions based on new size
            slides.forEach((slide, index) => {
                slide.style.left = slideSize * index + 'px';
            });
            
            // Re-center the track on the active slide after resize
            const activeSlide = getActiveSlide();
            if (activeSlide) {
                 track.style.transform = 'translateX(-' + activeSlide.style.left + ')';
            }
        });
        
    });
});