/**
 * worship.js
 * Lightweight JavaScript for IntersectionObserver reveal-on-scroll
 * and smooth scrolling for anchor links.
 */

document.addEventListener('DOMContentLoaded', () => {
    // 1. IntersectionObserver for Reveal-on-Scroll
    const revealElements = document.querySelectorAll('.fade-in, .slide-up, .reveal-card, .reveal-fade-in');
    
    const observerOptions = {
        root: null, // viewport
        rootMargin: '0px',
        threshold: 0.1 // Reveal when 10% of element is visible
    };

    const observer = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const element = entry.target;
                const parent = element.closest('.reveal-stagger');

                if (parent) {
                    // Handle stagger reveal for elements inside a container
                    const staggeredElements = parent.querySelectorAll('.fade-in, .slide-up, .reveal-card, .reveal-fade-in');
                    staggeredElements.forEach((staggeredEl, index) => {
                        const delayAttr = staggeredEl.getAttribute('data-delay');
                        const delay = delayAttr ? parseFloat(delayAttr.replace('s', '')) : (index * 0.1);
                        
                        setTimeout(() => {
                            staggeredEl.classList.add('visible');
                        }, delay * 1000);
                    });
                    
                    // Stop observing the parent container once its children are triggered
                    observer.unobserve(parent);

                } else if (!element.closest('.reveal-stagger')) {
                    // Handle non-staggered individual elements
                    const delayAttr = element.getAttribute('data-delay');
                    const delay = delayAttr ? parseFloat(delayAttr.replace('s', '')) : 0;
                    
                    setTimeout(() => {
                        element.classList.add('visible');
                    }, delay * 1000);
                    
                    // Stop observing the element
                    observer.unobserve(element);
                }
            }
        });
    }, observerOptions);

    // Observe all elements that need a reveal effect
    // We observe the staggered containers only once, and individual elements otherwise
    const observedContainers = new Set();
    revealElements.forEach(el => {
        const parent = el.closest('.reveal-stagger');
        if (parent && !observedContainers.has(parent)) {
            observer.observe(parent);
            observedContainers.add(parent);
        } else if (!parent) {
            observer.observe(el);
        }
    });

    // 2. Smooth scroll for anchor links
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            // Check if the hash is not just '#'
            if (this.getAttribute('href') !== '#') {
                e.preventDefault();
                document.querySelector(this.getAttribute('href')).scrollIntoView({
                    behavior: 'smooth'
                });
            }
        });
    });
});